
///////////////////////////////////////////////////////////////////////////////
//
//   ROCKWELL SEMICONDUCTOR SYSTEMS - COMMERCIAL GPS BUSINESS
//
///////////////////////////////////////////////////////////////////////////////
//
//
//   MSG1000.C - Message 1000 Processing
// 
//
//   DESCRIPTION
//
//   Functions to decode, build, and display Zodiac 1000 messages.
//
//
//   REVISION HISTORY
//
//   $Log:   V:\Projects\Labmon\Source\Archives\MSG1000.C_v  $
//   
//      Rev 1.22   07 Jan 2000 14:59:06   phungh
//   Labmon60 : Rounding up the UTCSecond
//   base on the UTCnanosecond value
//   
//      Rev 1.21   08 Dec 1999 16:15:52   phungh
//   Fix bug when monitoring the continuity of
//   NAV status ( the feature enabled with 
//   extract COV data)
//   
//      Rev 1.20   15 Dec 1998 14:27:08   phungh
//   Labmon53 : Fix flash download in "slow
//   mode" ,and record "in/out of NAV"
//   occurance using "xtract COV"
//   
//      Rev 1.19   11 Nov 1998 11:52:14   phungh
//   lbmon522 : Add logging the EVPE to file
//   "cov.txt" when the EVPE value is crossing
//   the set value in either direction.  Changes 
//   made to labmon.c and msg1000.c

//      Rev 1.17   11 Nov 1998 10:37:54   phungh
//   lbmon521: Log the EHPE to file "cov.txt"
//   when the EHPE value passing the set 
//   value in either direction.  Change made to
//   labmon.c and msg1000.c.
//   
//      Rev 1.11   Jul 06 1998 09:08:40   BANHND
//   added more DR features
//   
//      Rev 1.8   Jul 09 1997 09:50:22   COLEJ
//   Multiple Changes...
//   
//      Rev 1.7   Feb 25 1997 14:35:16   COLEJ
//   changed method of calculating delta positions
//   using a matrix vice a constant
//   
//      Rev 1.6   Feb 12 1997 16:05:14   COLEJ
//    
//   
//      Rev 1.5   Oct 31 1996 11:28:42   COLEJ
//    
//   
//      Rev 1.4   Aug 26 1996 18:42:50   COLEJ
//    
//   
//      Rev 1.3   Aug 15 1996 16:53:54   COLEJ
//   Added option to remove sign from LAT and LON on LLA extraction.
//   
//      Rev 1.2   Aug 14 1996 18:14:26   COLEJ
//   corrected LLA and COV extract GPSTimeSec output.
//   
//      Rev 1.1   Aug 14 1996 09:09:48   COLEJ
//   Removed UTCNSECS field and moved up other fields.
//   Changed GPS field to display seconds vice time.
//   Added ability to select units for SPD field.
//   Added msg1108 processing.
//   Changed UTC TIME processing to hh:mm:ss, and conditionally.
//   
//   
//      Rev 1.0   13 May 1996 14:52:38   GPSADMIN
//   Initial release to version control.
//
//
////////////////////////////////////////////////////////////////////////////////
 
#include <stdio.h>
#include <graph.h>
#include <string.h>
#include <math.h>

#include "gentypes.h"
#include "gencons.h"
#include "ztypes.h"
#include "zcons.h"   
#include "util.h" 
#include "labmon.h"
#include "display.h"

extern unsigned long CntNav, CntPnt;

void show_filter(short);
char    NavigationMode[6];
unsigned long  WONCount = 0; // used for counting power cycles
extern unsigned long  Cnt1000;
char Buffer[80];
static tBOOL EHPEMaxExcd = 1;
static tBOOL EVPEMaxExcd = 1;

////////////////////////////////////////////////////////////////////////////////
//   
//   Decode a Zodiac 1000 message from the ZMsgBuf.
//
void Dec1000(tMSGBUF *ZMsgBuf, tMSG1000 *Msg)
{ 
   // perform special decoding to unpack words
   GetBit(&SolutionStatusBits.InvalidAltUsed  ,  ZMsgBuf, 10, 0);  // 1=true
   GetBit(&SolutionStatusBits.InvalidNoDGPS   ,  ZMsgBuf, 10, 1);  // 1=true
   GetBit(&SolutionStatusBits.InvalidNumTrk   ,  ZMsgBuf, 10, 2);  // 1=true 
   GetBit(&SolutionStatusBits.InvalidExcEHPE  ,  ZMsgBuf, 10, 3);  // 1=true
   GetBit(&SolutionStatusBits.InvalidExcEVPE  ,  ZMsgBuf, 10, 4);  // 1=true
   GetBit(&SolutionStatusBits.InvalidNoDRMeas ,  ZMsgBuf, 10, 5);  // 1=true
   GetBit(&SolutionStatusBits.InvalidNoConGPS ,  ZMsgBuf, 10, 6);  // 1=true
   GetBit(&SolutionStatusBits.InvalidNoGpsOnly,  ZMsgBuf, 10, 7);  // 1=true

   GetBit(&SolutionStatusBits.TypePropagated, ZMsgBuf, 11, 0);  // 1=propagated
   GetBit(&SolutionStatusBits.TypeAltUsed   , ZMsgBuf, 11, 1);  // 1=altitude used
   GetBit(&SolutionStatusBits.TypeDGPS      , ZMsgBuf, 11, 2);  // 1=DGPS mode
   GetBit(&SolutionStatusBits.TypePower     , ZMsgBuf, 11, 3);  // 1=power off
   GetBit(&SolutionStatusBits.TypeGPS       , ZMsgBuf, 11, 4);  // 1=GPS only
   GetBit(&SolutionStatusBits.TypeConGPS    , ZMsgBuf, 11, 5);  // 1=GPS and DR
   GetBit(&SolutionStatusBits.TypeDRCal     , ZMsgBuf, 11, 6);  // 1=DR using stored calibration data
   GetBit(&SolutionStatusBits.TypeZeroSpeed , ZMsgBuf, 11, 7);  // 1=Zero Speed calibration
}                                      
                                     
////////////////////////////////////////////////////////////////////////////////
//   
//   Build a Zodiac 1000 message in the ZMsgBuf. 
//
void Bld1000(tMSGBUF *ZMsgBuf, tMSG1000 *Msg)
{
   // ack flags assumed preset

   PutShort  (0x81FF                       , ZMsgBuf, 1);      // message preamble
   PutShort  (1000                         , ZMsgBuf, 2);      // message number
   PutShort  (49                           , ZMsgBuf, 3);      // message length
 //PutShort  (CheckSum(ZMsgBuf[0], 4)      , ZMsgBuf, 5);      // header check sum

   PutBit(SolutionStatusBits.InvalidAltUsed,  ZMsgBuf, 10, 0);  // 1=true
   PutBit(SolutionStatusBits.InvalidNoDGPS ,  ZMsgBuf, 10, 1);  // 1=true
   PutBit(SolutionStatusBits.InvalidNumTrk ,  ZMsgBuf, 10, 2);  // 1=true 
   PutBit(SolutionStatusBits.InvalidExcEHPE,  ZMsgBuf, 10, 3);  // 1=true
   PutBit(SolutionStatusBits.InvalidExcEVPE,  ZMsgBuf, 10, 4);  // 1=true
   PutBit(SolutionStatusBits.InvalidNoDRMeas, ZMsgBuf, 10, 5);  // 1=true
   PutBit(SolutionStatusBits.InvalidNoConGPS, ZMsgBuf, 10, 6);  // 1=true
   PutBit(SolutionStatusBits.InvalidNoGpsOnly,ZMsgBuf, 10, 7);  // 1=true

   PutBit(SolutionStatusBits.TypePropagated, ZMsgBuf, 11, 0);  // 1=propagated
   PutBit(SolutionStatusBits.TypeAltUsed   , ZMsgBuf, 11, 1);  // 1=altitude used
   PutBit(SolutionStatusBits.TypeDGPS      , ZMsgBuf, 11, 2);  // 1=DGPS mode
   PutBit(SolutionStatusBits.TypePower     , ZMsgBuf, 11, 3);  // 1=power off
   PutBit(SolutionStatusBits.TypeGPS       , ZMsgBuf, 11, 4);  // 1=GPS only
   PutBit(SolutionStatusBits.TypeConGPS    , ZMsgBuf, 11, 5);  // 1=GPS and DR
   PutBit(SolutionStatusBits.TypeDRCal     , ZMsgBuf, 11, 6);  // 1=DR using stored calibration data
   PutBit(SolutionStatusBits.TypeZeroSpeed , ZMsgBuf, 11, 7);  // 1=Zero Speed calibration
    
 //PutShort  (CheckSum(ZMsgBuf, 6, 49)      , ZMsgBuf, 55);    // data check sum
}

////////////////////////////////////////////////////////////////////////////////
//   
//   Display a Zodiac 1000 message from the ZMsgBuf.
//
void Show1000(tMSG1000 *Msg)
{
   char    Buf[80] = "";
   char    *DayName[7] ={"SUN", "MON", "TUE", "WED", "THU", "FRI", "SAT"}; 
   char    UTCDayMonthYear[15],UTCHourMinSec[15]; 
   char    DayOfWeek[5];   
   char    SolutionInvalid[9], SolutionType[9];        
//   short   GPSHours, GPSMinutes, GPSSeconds;
   double  GPSTimeSeconds,GPSTimeNSeconds;
   unsigned long  tempsecs;
   unsigned short gpsprecision = 7;  // used to calculate precision field of gpssec output
   
   float   GroundSpeed,gspd, SpeedOverGround;  //gspd is used in ground speed unit coversions/output
   float   TrueCourse;  
   float   ClimbRate;
    
   char    LatOut[15], LonOut[15], AltOut[15];
   double  Latitude, Longitude, Height;   
//   double  dlat, dlon, dalt;  // used for delta positions in lla extraction ???

   // used for calulating delta positions
   tGeoPos GeoPos, RefGeoPos; // strutures for lat/lon/alt in degrees
   double  ENU[3];            // used for delta positions out in meters 
   extern double ref_lat, ref_lon, ref_alt;  // in degrees
   extern double lat, lon, alt;  // where is this used
   extern short  delta; 

   
//   double  DeltaTime = 0;
//   static double AlongTrackAccel = 0, CrossTrackAccel = 0, VerticalAccel = 0;
//   static double AngularRate = 0; 
//   static double GPSTimeSecondsLast = 0;
//   static double TrueCourseLast = 0, ClimbRateLast = 0, GroundSpeedLast = 0;
  
   
   extern unsigned short StatMask;   
   extern float  maxpdop, maxhdop, maxvdop;
   extern short  minsats;
   extern short  filter_on; 
   extern short  xtract;  
   extern char   xtractid[4];
   extern short  spdunits;
   extern tBOOL  MSG1108ON;
   extern tBOOL  ABS_LLA; 
   
   extern FILE   *stream3;
   
   unsigned long WPercent;
   
   // convert values
   Latitude        = Msg->Latitude    / 1E+08 * R2D;    // Latitude in degrees
   Longitude       = Msg->Longitude   / 1E+08 * R2D;    // Longitude in degrees   
   Height          = Msg->Height      / 1E+02;
   TrueCourse      = (float) (Msg->TrueCourse  / 1E+03);
   GroundSpeed     = (float) (Msg->GroundSpeed / 1E+02);
   gspd            = GroundSpeed;    
   SpeedOverGround = (float) (GroundSpeed * MPS2KN);    
   ClimbRate       = (float) (Msg->ClimbRate    /1E+02);
                                                      
   // perform special processing for output 
   lat = Msg->Latitude    / 1E+08;   // lat in radians
   lon = Msg->Longitude   / 1E+08;   // lon in radians
   alt = Msg->Height      / 1E+02;
   
   // convert GroundSpeed to units requested in cfg file - default is m/s
   gspd = GroundSpeed;
   if(spdunits == 1) gspd /= (float) 0.44704;  // convert m/s to MPH
   if(spdunits == 2) gspd *= (float) 3.6;      // convert m/s to KPH 

   // navigation validity  
   strcpy(SolutionInvalid, "        ");
   if(SolutionStatusBits.InvalidAltUsed  ) SolutionInvalid[0] = 'A';
   if(SolutionStatusBits.InvalidNoDGPS   ) SolutionInvalid[1] = 'D';
   if(SolutionStatusBits.InvalidNumTrk   ) SolutionInvalid[2] = 'N';
   if(SolutionStatusBits.InvalidExcEHPE  ) SolutionInvalid[3] = 'H';
   if(SolutionStatusBits.InvalidExcEVPE  ) SolutionInvalid[4] = 'V';
   if(SolutionStatusBits.InvalidNoDRMeas ) SolutionInvalid[5] = 'M';
   if(SolutionStatusBits.InvalidNoConGPS ) SolutionInvalid[6] = 'C';
   if(SolutionStatusBits.InvalidNoGpsOnly) SolutionInvalid[7] = 'G';

   // navigation solution type and mode
   strcpy(SolutionType, "        "); 
   strcpy(NavigationMode, "  NAV");  
   
   if(SolutionStatusBits.TypePropagated ){
      SolutionType[0]    = 'P';
   }
   if(SolutionStatusBits.TypeAltUsed    ){
      SolutionType[1]    = 'A';
   }    
   if(SolutionStatusBits.TypePower      ){
      SolutionType[3]    = 'W'; 
      WONCount++;
   }
   if(SolutionStatusBits.TypeGPS        ){
      SolutionType[4]    = 'G'; 
   }
   if(SolutionStatusBits.TypeConGPS     ){
      SolutionType[5]    = 'C'; 
   }
   if(SolutionStatusBits.TypeDRCal      ){
      SolutionType[6]    = 'D'; 
   }
   if(SolutionStatusBits.TypeZeroSpeed  ){
      SolutionType[7]    = 'Z'; 
   }

   
   // calculate percent time W is on
   if(Cnt1000) WPercent = (WONCount * 100)/Cnt1000;
   else WPercent = 0;
    
   // navigation mode
   if(strcmp(SolutionInvalid, "        ") != 0){
      strcpy(NavigationMode, "  ACQ");
     
  //    AlongTrackAccel = 0;
  //    VerticalAccel   = 0;  
  //    CrossTrackAccel = 0;
   }
   else{

     if(SolutionStatusBits.TypeDGPS       ){
       SolutionType[2]    = 'D'; 
       strcpy(NavigationMode, " DGPS");
     }

      // derive accelerations from valid course and speed
  //    DeltaTime        = Msg->GPSTimeSeconds - GPSTimeSecondsLast; 
  //    if(DeltaTime >= 1.f){     
  //       AngularRate      = (TrueCourse  - TrueCourseLast)  / DeltaTime;      
  //       AlongTrackAccel  = (GroundSpeed - GroundSpeedLast) / DeltaTime;
  //       VerticalAccel    = (ClimbRate   - ClimbRateLast)   / DeltaTime;
  //       CrossTrackAccel  =-AngularRate * GroundSpeed;      
  //    }
  //    GPSTimeSecondsLast = Msg->GPSTimeSeconds;
  //    TrueCourseLast  = TrueCourse;
  //    GroundSpeedLast = GroundSpeed;
  //    ClimbRateLast   = ClimbRate;
   }
   
   // time values  -  only displaying gps seconds jc7-29-96
//   GPSHours    = (short) (Msg->GPSTimeSeconds / 3600.f);
//   TimeTemp    = Msg->GPSTimeSeconds - GPSHours * 3600.f;
//   GPSMinutes  = (short) (TimeTemp  / 60.f);
//   TimeTemp    = TimeTemp - GPSMinutes * 60.f;
//   GPSSeconds  = (short) TimeTemp;  
   
   
   // calculate gpssec field
   tempsecs = Msg->GPSTimeSeconds;
   while( tempsecs /= 10 ){ gpsprecision--; };
   // get rid of roundoff for nano seconds
   GPSTimeNSeconds = (double) ( Msg->GPSTimeNanoseconds / (unsigned long) pow( 10.0, (9.0 - (double) gpsprecision) ) );
   GPSTimeNSeconds *= pow( 10.0, (9.0 - (double) gpsprecision) );
   GPSTimeSeconds = Msg->GPSTimeSeconds + GPSTimeNSeconds / 1E+09;

   
   sprintf(DayOfWeek,"%s",
           DayName[(short)(Msg->GPSTimeSeconds / 86400)]);
   sprintf(UTCDayMonthYear,"%02d/%02d/%02d",
           Msg->UTCMonth,Msg->UTCDay,(Msg->UTCYear % 100));

   if(Msg->UTCNanoseconds > 500000000)
   	  Msg->UTCSeconds++;

   sprintf(UTCHourMinSec,"%02d:%02d:%02d",
           Msg->UTCHours,Msg->UTCMinutes,Msg->UTCSeconds);  
//   sprintf(GPSHourMinSecNsec,"%3d:%02d:%02d.%03d",
//           GPSHours,GPSMinutes,GPSSeconds,               
//           Msg->GPSTimeNanoseconds / 1000000);  

   // get delta positions when in delta or xtract modes
   if (delta || xtract){
      // load tGeoPos structs with terms in degrees
      GeoPos.lat = Latitude; 
      GeoPos.lon = Longitude;
      GeoPos.alt = Height;
      RefGeoPos.lat = ref_lat;
      RefGeoPos.lon = ref_lon;
      RefGeoPos.alt = ref_alt;
      
      GeoidToENU( GeoPos, RefGeoPos, ENU );  // call function to calculate delta's
   
      ENU[2] -= ref_alt;  // this converts the change in altitude because ENU[2] is maintained stationary
   }
   
   
   // normal or delta position output mode  
   if(!delta){ 
      // convert lat, lon to degrees/minutes string
      ConvLatDegDM(Latitude,  LatOut);
      ConvLonDegDM(Longitude, LonOut);
      sprintf(AltOut,"%12.2f", Height); 
   }
   else{
      // output delta lat/lon/alt in meters from reference position
      sprintf(LatOut,"%12.2f ", ENU[1]);
      sprintf(LonOut,"%12.2f ", ENU[0]); 
      sprintf(AltOut,"%12.2f" , ENU[2]);   //Height - ref_alt);
   } 
   
   // filter the data using binary mode parameters
   CntPnt++;
         
   if(((StatMask & Msg->SolutionValidity) == 0) &&      
      (Msg->NumMeasUsed   >= (unsigned short) minsats)           ){       
               
      filter_on = 0; 
      CntNav++;
      
      // write extracted data to a file
      if(xtract){
         if(strcmp(xtractid,"LLA") == 0){     

            //dlon = ENU[0];    
            //dlat = ENU[1];    
            //dalt = ENU[2];   //alt - ref_alt;      
         
            if( ABS_LLA ) // wants to remove signs on lat log for plotting
               fprintf(stream3, "%16.9lf %11.6f %11.6f %9.2f %10.2f %11.2f %9.2f\r\n",      
                  GPSTimeSeconds, fabs(Longitude), fabs(Latitude), Height, ENU[0], ENU[1], ENU[2]);
            else          // wants to leave signs on lat log like displayed values
               fprintf(stream3, "%16.9lf %11.6f %11.6f %9.2f %10.2f %11.2f %9.2f\r\n",      
                  GPSTimeSeconds, Longitude, Latitude, Height, ENU[0], ENU[1], ENU[2]);
		 }
      }                                
   }      
   else{      
      filter_on = 1;      
   }      
    
   // indicate when filtering is occurring 
   show_filter(filter_on);


   if(xtract){
      if(strcmp(xtractid,"COV") == 0){ 
         if( ((SolutionInvalid[3] == 'H')||(SolutionInvalid[4] == 'V')) &&
              				   						 (EHPEMaxExcd == 1) ) {
//         if( ((SolutionInvalid[3] == 'H')&&(EHPEMaxExcd == 1)) || 
//             ((SolutionInvalid[4] == 'V')&&(EVPEMaxExcd == 1)) ) {
            
         //check for whether the above 'if' was entered because of 'H' or 'V' criteria exceeded
//			if( (SolutionInvalid[3] == 'H')&&(EHPEMaxExcd == 1) )
            	EHPEMaxExcd = 0;     
//			if( (SolutionInvalid[4] == 'V')&&(EVPEMaxExcd == 1) )
//            	EVPEMaxExcd = 0;     

            fprintf(stream3, 
            	    "%16.9f %9.2G %9.2G %10s %10s %10s %10s \n",      
//            "%16.9f \t%9.2G \t%9.2G \t%9.2G \t%9.2G \t%10.0f \t%9.2G \t%10.0f \t%9.2G\n",      
                    GPSTimeSeconds, 
                    Msg->EHPE / 1E+02,
                    Msg->EVPE / 1E+02,
                    UTCDayMonthYear,
                    UTCHourMinSec,
                    Buffer,
                    NavigationMode);
		 }
		 else if( (SolutionInvalid[3] == ' ')&&
		 		  (SolutionInvalid[4] == ' ')&&
		 		           (EHPEMaxExcd == 0) )  {
//		 else if( ((SolutionInvalid[3] == ' ')&&(EHPEMaxExcd == 0)) ||
//		 		  ((SolutionInvalid[4] == ' ')&&(EVPEMaxExcd == 0)) ) {

         //check for whether the above 'else if' was entered because of 'H' or 'V' criteria was met
//			if( (SolutionInvalid[3] == ' ')&&(EHPEMaxExcd == 0) )
            	EHPEMaxExcd = 1;     
//			if( (SolutionInvalid[4] == ' ')&&(EVPEMaxExcd == 0) )
//            	EVPEMaxExcd = 1;     

            fprintf(stream3, 
            	    "%16.9f %9.2G %9.2G %10s %10s %10s %10s \n",      
//            "%16.9f \t%9.2G \t%9.2G \t%9.2G \t%9.2G \t%10.0f \t%9.2G \t%10.0f \t%9.2G\n",      
                    GPSTimeSeconds, 
                    Msg->EHPE / 1E+02,
                    Msg->EVPE / 1E+02,
                    UTCDayMonthYear,
                    UTCHourMinSec,
                    Buffer,
                    NavigationMode);
         }
      }
   }         	



   // output the data items   
   sprintf(Buf,"%05d"  ,Msg->SequenceNumber          );ShowText(Buf,TIMR+ 6,TIMC+ 4);
   sprintf(Buf,"%05d"  ,Msg->MeasSequenceNumber      );ShowText(Buf,TIMR+ 6,TIMC+12);
   sprintf(Buf,"%05s"  ,NavigationMode               );ShowText(Buf,MODR   ,MODC+ 5);
   sprintf(Buf,"%08s"  ,SolutionInvalid              );ShowText(Buf,MODR+ 1,MODC+ 5);
   sprintf(Buf,"%08s"  ,SolutionType                 );ShowText(Buf,MODR+ 2,MODC+ 4);
   sprintf(Buf,"%4d"   ,Msg->NumMeasUsed             );ShowText(Buf,DOPR- 2,DOPC+ 6);
   sprintf(Buf,"%1d"   ,Msg->PolarNav & 0x1          );ShowText(Buf,GEOR+ 2,GEOC+10);
   sprintf(Buf,"%4d "  ,Msg->GPSWeek                 );ShowText(Buf,TIMR+ 4,TIMC+ 5);
   sprintf(Buf,"%9.*f" ,gpsprecision,GPSTimeSeconds  );ShowText(Buf,TIMR+ 2,TIMC+ 8);
   sprintf(Buf,"%09ld" ,Msg->GPSTimeNanoseconds      );ShowText(Buf,TIMR+ 3,TIMC+ 8);
   sprintf(Buf,"%s"    ,DayOfWeek                    );ShowText(Buf,TIMR+ 4,TIMC+14);
   sprintf(Buf,"%s"    ,UTCDayMonthYear              );ShowText(Buf,TIMR   ,TIMC+ 9);                 
   if(!MSG1108ON){sprintf(Buf,"  %s"  ,UTCHourMinSec );ShowText(Buf,TIMR+ 1,TIMC+ 7);}
//   sprintf(Buf,"%09ld" ,Msg->UTCNanoseconds          );ShowText(Buf,TIMR+ 2,TIMC+ 8);   taking out jc7-29-96
   sprintf(Buf,"%s"    ,LatOut                       );ShowText(Buf,LATR   ,LATC+ 4);
   sprintf(Buf,"%s"    ,LonOut                       );ShowText(Buf,LATR+ 1,LATC+ 4);
   sprintf(Buf,"%s"    ,AltOut                       );ShowText(Buf,LATR+ 2,LATC+ 4);
   sprintf(Buf,"%7.0f" ,Msg->GeoidalSeparation /1E+02);ShowText(Buf,GEOR   ,GEOC+ 4); 
   sprintf(Buf,"%6.2f" ,gspd                         );ShowText(Buf,SPDR   ,SPDC+ 4); 
   sprintf(Buf,"%6.2f" ,ClimbRate                    );ShowText(Buf,SPDR+ 1,SPDC+ 4); 
   sprintf(Buf,"%6.2f" ,SpeedOverGround              );ShowText(Buf,SOGR   ,SOGC+ 4); 
   sprintf(Buf,"%5.1f",TrueCourse             *R2D  );ShowText(Buf,SOGR+ 1,SOGC+ 5);  
//   sprintf(Buf,"%6.2f" ,AlongTrackAccel              );ShowText(Buf,VELR   ,SPDC+ 4); 
//   sprintf(Buf,"%6.2f" ,CrossTrackAccel              );ShowText(Buf,VELR+ 1,SPDC+ 4); 
//   sprintf(Buf,"%6.2f" ,VerticalAccel                );ShowText(Buf,VELR+ 2,SPDC+ 4);
   sprintf(Buf,"%6.2f" ,Msg->MagneticVariation /1E+04);ShowText(Buf,SOGR+ 2,SOGC+ 4); 
   sprintf(Buf,"%7.2f" ,(float)(Msg->PolarNav>>1)/100.0);ShowText(Buf,SOGR+3,SOGC+5);
   sprintf(Buf,"%3d"   ,Msg->MapDatum                );ShowText(Buf,GEOR+ 1,GEOC+ 8); 
   sprintf(Buf,"%9.2G" ,Msg->EHPE              /1E+02);ShowText(Buf,HERR   ,HERC+ 5); 
   sprintf(Buf,"%9.2G" ,Msg->EVPE              /1E+02);ShowText(Buf,HERR+ 1,HERC+ 5); 
   sprintf(Buf,"%9.2G" ,Msg->ETE               /1E+02);ShowText(Buf,HERR+ 2,HERC+ 5); 
   sprintf(Buf,"%9.2G" ,Msg->EHVE              /1E+02);ShowText(Buf,HERR+ 3,HERC+ 5); 
   sprintf(Buf,"%10.0f",Msg->ClockBias         /1E+02);ShowText(Buf,CBER   ,CBEC+ 4); 
   sprintf(Buf,"%9.2G" ,Msg->ClockBiasStdDev   /1E+02);ShowText(Buf,CBER+ 1,CBEC+ 5); 
   sprintf(Buf,"%10.0f",Msg->ClockDrift        /1E+02);ShowText(Buf,CBER+ 2,CBEC+ 4); 
   sprintf(Buf,"%9.2G" ,Msg->ClockDriftStdDev  /1E+02);ShowText(Buf,CBER+ 3,CBEC+ 5);                                    
   sprintf(Buf,"%3lu%%",WPercent                     );ShowText(Buf,PRTR+ 2,PRTC+ 5);                                    

}
